﻿<#
.SYNOPSIS
    installer - Installation script for all updates included in package
.DESCRIPTION
    Script runs through all files and starts install command for each based on the file specifics
#>

$sriptPath = Split-Path $MyInvocation.MyCommand.Path
$sriptName = $MyInvocation.MyCommand.Name
$workDir = Join-Path -Path $env:ProgramData -ChildPath "colibri"
$updatesDir = Join-Path -Path $workDir -ChildPath "updates"
$scanResultFile = Join-Path -Path $updatesDir -ChildPath "MissingUpdates.json"
$logDir = Join-Path -Path $workDir -ChildPath "logs"
$scriptLogFile = "Updates_Install_{0}.log" -f $(Get-Date -Format yyyyMMddTHHmmss)
$LogFilePath = Join-Path -Path $logDir -ChildPath $scriptLogFile
$jsonLineTemplate='{"FileName": "XXX", "ExitCode": "YYY"}'


Function Write-Log {
        [CmdletBinding()]
        Param (

                [Parameter(Mandatory=$true,Position=0)]
                [Alias('Text')]
                [string[]]$Message,

                [Parameter(Mandatory=$false,Position=1)]
                [ValidateSet('Normal','Warning','Error','Debug')]
                [string]$Severity = 'Normal'
        )

        [string]$LogTime = (Get-Date -Format 'HH:mm:ss.fff').ToString()
        [string]$LogDate = (Get-Date -Format 'MM-dd-yyyy').ToString()
        [string]$LogLine = "[$LogDate $LogTime]"

        Switch ($Severity) {
                'Normal'        { $LogLine = "$LogLine [INFO] :: $Message" }
                'Warning'       { $LogLine = "$LogLine [WARNING] :: $Message" }
                'Error'         { $LogLine = "$LogLine [ERROR] :: $Message" }
                'Debug'         { $LogLine = "$LogLine [DEBUG] :: $Message" }
        }

        # Check existance of the log dir
        If (-not (Test-Path -LiteralPath $logDir -PathType 'Container')) { $null = New-Item -Path $logDir -Type 'Directory' -Force -ErrorAction 'Stop' }

        [string]$LogFilePath = Join-Path -Path $logDir -ChildPath $scriptLogFile

        Try {
                # Write log line to the file
                $LogLine | Out-File -FilePath $LogFilePath -Append -NoClobber -Force -Encoding 'UTF8' -ErrorAction 'Stop'
        }
        Catch {
                Write-Host -Object "[$LogDate $LogTime] [$ScriptStage] [ERROR] :: Failed to write message [$Message] to the log file [$LogFilePath].[$($_.Exception)]]" -ForegroundColor  'Red'
        }
}

$FinalExitCode=0
$ClearScanResult=$false
$arrAllowedExitCodes = @(0,1641,3010)

$valueToCheck = 3
$ResJson = ""
$ContentCsv = Import-CSV "$($sriptpath)\content.csv"
$UpdatesList = Get-ChildItem "$($sriptpath)\*" -Include "*.exe","*.cab","*.msu"


foreach ($Upd in $UpdatesList) {
    switch ($Upd.Extension) {
        ".exe" { 
            if ($Upd.Name -match "sql") {
                $ArgumentList = "/action=patch /AllInstances /quiet /IAcceptSQLServerLicenseTerms"
            } elseif ($Upd.Name -match "kb890830") {
                $ArgumentList = "/quiet"
            } else {
                $ArgumentList = "/install /quiet /norestart"
            }
            $processname = "`"$($upd.FullName)`""
        }
        ".cab" {
            $csvitem = $ContentCsv | Where-Object {$_.'filename' -eq "$($upd.Name)"}
            If ($($csvitem.'categories').ToLower() -like '*office*') {
                $tmpdir = "$($sriptpath)\tmp"
                if (!(Test-Path "$tmpdir")) {
                    New-Item -Path "$tmpdir" -ItemType Directory -Force | Out-Null
                } else {
                    Get-ChildItem -Path "$tmpdir" -Recurse | Remove-Item -force -recurse
                }
                Start-Process -FilePath "cmd.exe" -ArgumentList "/c expand.exe `"$($upd.FullName)`" -f:* `"$tmpdir`" > nul 2>&1" -Wait -WindowStyle Hidden | Out-Null
                #expand "$($Update.FullName)" -F:*.msp "$tmpdir"
                $msp = (Get-Item "$tmpdir\*.msp").FullName
                $ArgumentList = "/p `"$msp`" /qr REBOOT=ReallySuppress MSIRESTARTMANAGERCONTROL=Disable"
                $processname = "msiexec"
             } else {
                $ArgumentList = "/Online /Add-Package /PackagePath:`"$($upd.FullName)`" /quiet /norestart"
                $processname = "DISM.exe" 
            }
        }
        ".msu" {
            $ArgumentList = "`"$($upd.FullName)`" /quiet /norestart"
            $processname = "wusa.exe"
        }
        #"msi" {
        #}
    }
    Write-Log "Installing $($Upd.Name)"
    Write-Log "Starting $($processname) with arguments $($ArgumentList)"
    $res = Start-Process $processname -Wait -WindowStyle Hidden -PassThru -ArgumentList $ArgumentList
    If (! ($arrAllowedExitCodes.Contains($res.ExitCode))) {
        $FinalExitCode=1
    } else {
        $ClearScanResult=$true
    }
    Write-Log "Finished with $($res.ExitCode) exitcode."

    $UpdJsonLine = $jsonLineTemplate.Replace('XXX', $Upd.Name).Replace('YYY',$res.ExitCode)
    $ResJson = $UpdJsonLine + ", " + $ResJson 
}
#$outputJson = "Updates_Install_{0}.json" -f $(Get-Date -Format yyyyMMddTHHmmss)
If ($ClearScanResult) {
    Write-Log "Removing previous updates scan file $scanResultFile"
    If (Test-Path $scanResultFile) {Remove-Item -Path $scanResultFile | Out-Null}
} 

$outputJson = "Updates_Install.json"
"[" +$ResJson.TrimEnd(", ") + "]" | Out-File "$($sriptpath)\$outputJson"

exit $FinalExitCode